import java.awt.Color;
import java.awt.image.BufferedImage;

import javax.swing.JLabel;

/* EscherFFT - ComplexImage.java
 * 
 * Author   : Nicolas Schoeni
 * Creation : 29 mai 2006
 * 
 * nicolas.schoeni@epfl.ch
 */

public class ComplexImageDoubleSize {
	public  float dataRe[][], dataIm[][];
	public  BufferedImage image; 
	public  int imageInMode;
	public  boolean inverse;
	public  final int size, size2, size2size2;
	public  float maxValue=1;
	public  float contrast=1;
	private ComplexImageSystem complexImageSystem;

	public ComplexImageDoubleSize(int size, boolean inverse, ComplexImageSystem complexImageSystem) {
		this(size, inverse, complexImageSystem, OutputSystem.createImage(size, Color.white));
	}
	public ComplexImageDoubleSize(int size, boolean inverse, ComplexImageSystem complexImageSystem, BufferedImage image) {
		this.image=image;
		this.complexImageSystem=complexImageSystem;
		this.size=size;
		size2=size*2;
		size2size2=size2*size2;
		this.inverse=inverse;
		dataRe = new float[size2][size2];
		dataIm = new float[size2][size2];
	}
	
	public void reset() {
		for (int i=0; i<size2; i++) {
			for (int j=0; j<size2; j++) {
				dataRe[i][j] = 0;
				dataIm[i][j] = 0;
			}
		}
	}
	
	public Object clone() {
		ComplexImage c = new ComplexImage(size, inverse, complexImageSystem);
		for (int i=0; i<size2; i++) {
			for (int j=0; j<size2; j++) {
				c.dataRe[i][j] = dataRe[i][j];
				c.dataIm[i][j] = dataIm[i][j];
			}
		}
		c.maxValue = maxValue;
		c.contrast = contrast;
		c.imageInMode = imageInMode;
		c.image.getGraphics().drawImage(image, 0, 0, null);
		return c;
	}

	public void load(ComplexImageDoubleSize other, int mode) {
		for (int i=0; i<size2; i++) {
			for (int j=0; j<size2; j++) {
				if (mode==ComplexColor.MODE_COMPLEX) {
					dataRe[i][j] = other.dataRe[i][j];
					dataIm[i][j] = other.dataIm[i][j];
				}
				else if (mode==ComplexColor.MODE_REAL_ONLY) {
					dataRe[i][j] = other.dataRe[i][j];
					dataIm[i][j] = 0;
				}
				else if (mode==ComplexColor.MODE_IM_ONLY) {
					dataRe[i][j] = 0;
					dataIm[i][j] = other.dataIm[i][j];
				}
				else if (mode==ComplexColor.MODE_MAGNITUDE) {
					dataRe[i][j] = other.dataRe[i][j]*other.dataRe[i][j]+other.dataIm[i][j]*other.dataIm[i][j];
					dataIm[i][j] = 0;
				}
				else if (mode==ComplexColor.MODE_PHASE) {
					double phi = Math.atan2(other.dataIm[i][j], other.dataRe[i][j]);
					dataRe[i][j] = (float)Math.cos(phi);
					dataIm[i][j] = (float)Math.sin(phi);
				}
			}
		}
	}
	
	public void load(ComplexImageDoubleSize other, int mode, int r) {
		int r2 = r*r/4;
		int s2 = size2/2;
		for (int i=0; i<size2; i++) {
			for (int j=0; j<size2; j++) {
				int i2 = ((i+s2)%size2)-s2; 
				int j2 = ((j+s2)%size2)-s2;
				if (i2*i2+j2*j2>r2) {
					dataRe[i][j] = 0;
					dataIm[i][j] = 0;
				}
				else if (mode==ComplexColor.MODE_COMPLEX) {
					dataRe[i][j] = other.dataRe[i][j];
					dataIm[i][j] = other.dataIm[i][j];
				}
				else if (mode==ComplexColor.MODE_REAL_ONLY) {
					dataRe[i][j] = other.dataRe[i][j];
					dataIm[i][j] = 0;
				}
				else if (mode==ComplexColor.MODE_IM_ONLY) {
					dataRe[i][j] = 0;
					dataIm[i][j] = other.dataIm[i][j];
				}
				else if (mode==ComplexColor.MODE_MAGNITUDE) {
					dataRe[i][j] = other.dataRe[i][j]*other.dataRe[i][j]+other.dataIm[i][j]*other.dataIm[i][j];
					dataIm[i][j] = 0;
				}
				else if (mode==ComplexColor.MODE_PHASE) {
					double phi = Math.atan2(other.dataIm[i][j], other.dataRe[i][j]);
					dataRe[i][j] = (float)Math.cos(phi);
					dataIm[i][j] = (float)Math.sin(phi);
				}
			}
		}
	}
	
	public void load(BufferedImage image) {
		//long t0 = System.currentTimeMillis();
		int w = image.getWidth();
		int h = image.getHeight();
		int off = size/2;
		float[] reIm = new float[2];
		for (int x=0; x<size2; x++) {
			for (int y=0; y<size2; y++) {
				//int i=(x+size+off)%size2, j=(y+size+off)%size2; 
				int i=(x+size2-w/2)%size2, j=(y+size2-h/2)%size2;
				if (x>=w||y>=h) {
					dataRe[i][j] = 0;
					dataIm[i][j] = 0;
				}
				else {
					int rgb = image.getRGB(x, y);
					ComplexColor.RGBToComplex(rgb, reIm);
					dataRe[i][j] = reIm[0];
					dataIm[i][j] = reIm[1];
					//System.out.println(dataRe[i][j]+" "+dataIm[i][j]);
				}
			}
		}
		//long t1 = System.currentTimeMillis();
		//System.out.println("load "+(t1-t0)+" ms");
	}

	public void load(int s, float[][] re, float[][] im) {
		//long t0 = System.currentTimeMillis();
		int off = size/2;
		for (int x=0; x<size2; x++) {
			for (int y=0; y<size2; y++) {
				int i=(x+size2-s/2)%size2, j=(y+size2-s/2)%size2;
				if (x>=s||y>=s) {
					dataRe[i][j] = 0;
					dataIm[i][j] = 0;
				}
				else {
					dataRe[i][j] = re[x][y];
					dataIm[i][j] = im[x][y];
				}
			}
		}
		//long t1 = System.currentTimeMillis();
		//System.out.println("load "+(t1-t0)+" ms");
	}
	
	public void selfOperation(int mode, int factor) {
		if (mode==ComplexColor.MODE_COMPLEX && factor==1) return;
		for (int i=0; i<size2; i++) {
			for (int j=0; j<size2; j++) {
				if (mode==ComplexColor.MODE_COMPLEX) {
					dataRe[i][j] *= (float)factor;
					dataIm[i][j] *= (float)factor;
				}
				else if (mode==ComplexColor.MODE_REAL_ONLY) {
					dataRe[i][j] *= (float)factor;
					dataIm[i][j] = 0;
				}
				else if (mode==ComplexColor.MODE_IM_ONLY) {
					dataRe[i][j] = 0;
					dataIm[i][j] *= (float)factor;
				}
				else if (mode==ComplexColor.MODE_MAGNITUDE) {
					float r = (float)Math.sqrt(dataRe[i][j]*dataRe[i][j]+dataIm[i][j]*dataIm[i][j]);
					dataRe[i][j] = r;
					dataIm[i][j] = 0;
				}
				else if (mode==ComplexColor.MODE_PHASE) {
					float r = (float)Math.sqrt(dataRe[i][j]*dataRe[i][j]+dataIm[i][j]*dataIm[i][j]);
					if (r==0) {
						dataRe[i][j] = 1;
						dataIm[i][j] = 0;
					}
					else {
						dataRe[i][j] /= (float)factor*r;
						dataIm[i][j] /= (float)factor*r;
					}
				}
			}
		}
		calculateMaxValue();
	}
	public void operation(ComplexImageDoubleSize other, int mode, int factor) {
		for (int i=0; i<size2; i++) {
			for (int j=0; j<size2; j++) {
				if (mode==ComplexColor.MODE_COMPLEX) {
					dataRe[i][j] += (float)factor*other.dataRe[i][j];
					dataIm[i][j] += (float)factor*other.dataIm[i][j];
				}
				else if (mode==ComplexColor.MODE_REAL_ONLY) {
					dataRe[i][j] += (float)factor*other.dataRe[i][j];
				}
				else if (mode==ComplexColor.MODE_IM_ONLY) {
					dataIm[i][j] += (float)factor*other.dataIm[i][j];
				}
				else if (mode==ComplexColor.MODE_MAGNITUDE) {
					float r = (float)Math.sqrt(other.dataRe[i][j]*other.dataRe[i][j]+other.dataIm[i][j]*other.dataIm[i][j]);
					dataRe[i][j] *= (float)factor*r;
					dataIm[i][j] *= (float)factor*r;
				}
				else if (mode==ComplexColor.MODE_PHASE) {
					double phi = Math.atan2(other.dataIm[i][j], other.dataRe[i][j]);
					dataRe[i][j] += (float)factor*(float)Math.cos(phi);
					dataIm[i][j] += (float)factor*(float)Math.sin(phi);
				}
			}
		}
		calculateMaxValue();
	}
	public void operation(int s, float[][] re, float[][] im, int mode, int factor) {
		int off = size/2;
		for (int x=0; x<size2; x++) {
			for (int y=0; y<size2; y++) {
				int i=(x+size2-s/2)%size2, j=(y+size2-s/2)%size2;
				if (x>=s||y>=s) {}
				else {
					if (mode==ComplexColor.MODE_COMPLEX) {
						dataRe[i][j] += (float)factor*re[x][y];
						dataIm[i][j] += (float)factor*im[x][y];
					}
					else if (mode==ComplexColor.MODE_REAL_ONLY) {
						dataRe[i][j] += (float)factor*re[x][y];
					}
					else if (mode==ComplexColor.MODE_IM_ONLY) {
						dataIm[i][j] += (float)factor*im[x][y];
					}
					else if (mode==ComplexColor.MODE_MAGNITUDE) {
						float r = (float)Math.sqrt(re[x][y]*re[x][y]+im[x][y]*im[x][y]);
						dataRe[i][j] *= (float)factor*r;
						dataIm[i][j] *= (float)factor*r;
					}
					else if (mode==ComplexColor.MODE_PHASE) {
						double phi = Math.atan2(im[x][y], re[x][y]);
						dataRe[i][j] += (float)factor*(float)Math.cos(phi);
						dataIm[i][j] += (float)factor*(float)Math.sin(phi);
					}
				}
			}
		}
		calculateMaxValue();
	}
	public void operation(int s, BufferedImage image, int mode, int factor) {
		float[] reIm = new float[2];
		int off = size/2;
		for (int x=0; x<size2; x++) {
			for (int y=0; y<size2; y++) {
				int i=(x+size2-s/2)%size2, j=(y+size2-s/2)%size2;
				if (x>=s||y>=s) {}
				else {
					int rgb = image.getRGB(x, y);
					ComplexColor.RGBToComplex(rgb, reIm);
					if (mode==ComplexColor.MODE_COMPLEX) {
						dataRe[i][j] += (float)factor*reIm[0];
						dataIm[i][j] += (float)factor*reIm[1];
					}
					else if (mode==ComplexColor.MODE_REAL_ONLY) {
						dataRe[i][j] += (float)factor*reIm[0];
					}
					else if (mode==ComplexColor.MODE_IM_ONLY) {
						dataIm[i][j] += (float)factor*reIm[1];
					}
					else if (mode==ComplexColor.MODE_MAGNITUDE) {
						float r = (float)Math.sqrt(reIm[0]*reIm[0]+reIm[1]*reIm[1]);
						dataRe[i][j] *= (float)factor*r;
						dataIm[i][j] *= (float)factor*r;
					}
					else if (mode==ComplexColor.MODE_PHASE) {
						double phi = Math.atan2(reIm[1], reIm[0]);
						dataRe[i][j] += (float)factor*(float)Math.cos(phi);
						dataIm[i][j] += (float)factor*(float)Math.sin(phi);
					}
				}
			}
		}
		calculateMaxValue();
	}

	public void doFFT() {
		//float_fft(size2, inverse, dataRe, dataIm);
		if (inverse) fft_back(size2, dataRe, dataIm);
		else fft(size2, dataRe, dataIm);
		calculateMaxValue();
	}
	
	public void setLabelValue(int x, int y, JLabel label) {
		if (x<0||y<0||x>=size||y>=size) {
			label.setText(" ");
		}
		else {
			int off = size/2;
			x=(x+size+off)%size2;
			y=(y+size+off)%size2;
			float re = dataRe[x][y];
			float im = dataIm[x][y];
			float a = (float)Math.sqrt(re*re+im*im);
			int phi = (int)Math.round(Math.atan2(im, re)*180f/Math.PI);
			re = Math.round(re*100f)/100f;
			im =  Math.round(im*100f)/100f;
			a =  Math.round(a*100f)/100f;
//			float[] hsb = new float[3];
//			ComplexColor.complexToHSB(re/maxValue, im/maxValue, hsb, mode);			
//			Color c = new Color(Color.HSBtoRGB(hsb[0], hsb[1], hsb[2]));
//			label.setForeground(c);
			label.setText("Re="+re+" Im="+im+" A="+a+" \u03c6="+phi+"");
		}
	}
	
	public void updateReIm2HSB() {
		//long t0 = System.currentTimeMillis();
		imageInMode = complexImageSystem.mode;
		complexImageSystem.lastHSB = this;
		int off = size/2;
		for (int y=0; y<size; y++) {
			for (int x=0; x<size; x++) {
				int x2 = (x+size+off)%size2;
				int y2 = (y+size+off)%size2;
				ComplexColor.complexToHSB(dataRe[x2][y2]/maxValue, dataIm[x2][y2]/maxValue, complexImageSystem.dataHSB[x][y], complexImageSystem.mode);
			}
		}
		//long t1 = System.currentTimeMillis();
		//System.out.println("ReIm->HSB "+(t1-t0)+" ms");
	}
	public void updateHSB2image() {
		//long t0 = System.currentTimeMillis();
		for (int x=0; x<size; x++) {
			for (int y=0; y<size; y++) {
				if (complexImageSystem.mode==ComplexColor.MODE_MAGNITUDE) {
					float bright = 1f-((1f-complexImageSystem.dataHSB[x][y][2])*contrast);
					if (bright>1) bright=1;
					if (bright<0) bright=0;
					int rgb = Color.HSBtoRGB(complexImageSystem.dataHSB[x][y][0], complexImageSystem.dataHSB[x][y][1], bright);
					image.setRGB(x, y, rgb);
				}
				else {
					float sat = complexImageSystem.dataHSB[x][y][1]*contrast;
					if (sat>1) sat=1;
					if (sat<0) sat=0;
					int rgb = Color.HSBtoRGB(complexImageSystem.dataHSB[x][y][0], sat, complexImageSystem.dataHSB[x][y][2]);
					image.setRGB(x, y, rgb);
				}
			}
		}
		//long t1 = System.currentTimeMillis();
		//System.out.println("HSB->image "+(t1-t0)+" ms");
	}
	
	public void setContrast(float contrast) {
		this.contrast=contrast;
		// check if hsb still valid
		if (complexImageSystem.lastHSB!=this) {
			updateReIm2HSB();
		}
		updateHSB2image();
	}

	public void calculateMaxValue() {
		maxValue=1;
		int off = size/2;
		for (int y=0; y<size; y++) {
			for (int x=0; x<size; x++) {
				int x2 = (x+size+off)%size2;
				int y2 = (y+size+off)%size2;
				if (dataRe[x2][y2]>maxValue) maxValue=dataRe[x2][y2];
				if (-dataRe[x2][y2]>maxValue) maxValue=-dataRe[x2][y2];
				if (dataIm[x2][y2]>maxValue) maxValue=dataIm[x2][y2];
				if (-dataIm[x2][y2]>maxValue) maxValue=-dataIm[x2][y2];
			}
		}
	}
	
	public void destroy() {
		if (image!=null) image.flush();
		image = null;
		dataRe=null;
		dataIm=null;
  }
	
	private final static double sTwoPi = 2.0 * Math.PI;
	
	private static void fft(int size, float[][] fr, float[][] fi)
	{
		float tr[]	= new float[size];
		float ti[]	= new float[size];
		int s14= size/4;
		int s34= size/2+s14;
		
		//long t0 = System.currentTimeMillis();

		for (int y=0; y<size; y++) {
			for (int x=0; x<size; x++) {
				fi[y][x] = -fi[y][x];
			}
		}
		
		// fft on the columns, use temporary storage
		for (int x=0; x<size; x++) {
			if (x>=s14&&x<s34) continue;
			for (int y=0; y<size; y++) {
				tr[y] = fr[y][x];
				ti[y] = fi[y][x];
			}
			float_fft1d(tr, ti, size);
			for (int y=0; y<size; y++) {
				fr[y][x] = tr[y];
				fi[y][x] = ti[y];
			}
		}
		
		//long t1 = System.currentTimeMillis();
		
		// fft on the image rows
		for (int y=0; y<size; y++) {
			float_fft1d(fr[y], fi[y], size);
		}

		for (int y=0; y<size; y++) {
			for (int x=0; x<size; x++) {
				fi[y][x] = -fi[y][x];
			}
		}
		//long t2 = System.currentTimeMillis();
		//System.out.println("FFT cols:"+(t1-t0)+" ms,  rows:"+(t2-t1)+" ms");
		
	}
	private static void fft_back(int size, float[][] fr, float[][] fi)
	{
		float tr[]	= new float[size];
		float ti[]	= new float[size];
		int s14= size/4;	
		int s34= size/2+s14;
		
		// negate im part if fft-1 
//		for (int y=0; y<size; y++) {
//			for (int x=0; x<size; x++) {
//				//fi[y][x] = -fi[y][x];
//			}
//		}
		
		// fft on the image rows
		for (int y=0; y<size; y++) {
			float_fft1d(fr[y], fi[y], size);
		}
		
		// fft on the columns, use temporary storage
		for (int x=0; x<size; x++) {
			if (x>=s14&&x<s34) continue;
			for (int y=0; y<size; y++) {
				tr[y] = fr[y][x];
				ti[y] = fi[y][x];
			}
			float_fft1d(tr, ti, size);
			for (int y=0; y<size; y++) {
				fr[y][x] = tr[y];
				fi[y][x] = ti[y];
			}
		}
		
		// negate back and rescale if fft-1 
		float scale = 1.0f / ((float)size * (float)size);
		for (int y=0; y<size; y++) {
			for (int x=0; x<size; x++) {
				fr[y][x] *= scale;
				//fi[y][x] *= -scale;
				fi[y][x] *= scale;
			}
		}
	}
	  
	  
	  private static void float_fft1d	(float xr [], float xi [], int n)
		{
//		----------------------------------------------------------------------------
	    int k, k0, k1, k2, k3, kinc, kinc2;
	    float qr, qi, rr, ri, sr, si, tr, ti, ur, ui;
	    float x1, w0r, w0i, w1r, w1i, w2r, w2i, w3r, w3i;
//		----------------------------------------------------------------------------
//	  radix 4 section
//		----------------------------------------------------------------------------
	    kinc = n;
//		----------------------------------------------------------------------------
	    while (kinc >= 4) {
	      kinc2	= kinc;
	      kinc	= kinc / 4;
//		----------------------------------------------------------------------------
	      for (k0 = 0; k0 < n; k0 += kinc2) {
	        k1 = k0 + kinc;
	        k2 = k1 + kinc;
	        k3 = k2 + kinc;
	        rr = xr[k0] + xr[k2];
	        ri = xi[k0] + xi[k2];
	        sr = xr[k0] - xr[k2];
	        si = xi[k0] - xi[k2];
	        tr = xr[k1] + xr[k3];
	        ti = xi[k1] + xi[k3];
	        ur = -xi[k1] + xi[k3];
	        ui = xr[k1] - xr[k3];
	        xr[k0] = rr + tr;
	        xi[k0] = ri + ti;
	        xr[k2] = sr + ur;
	        xi[k2] = si + ui;
	        xr[k1] = rr - tr;
	        xi[k1] = ri - ti;
	        xr[k3] = sr - ur;
	        xi[k3] = si - ui;
	      }
//		----------------------------------------------------------------------------
	      x1	= (float) sTwoPi / (float) kinc2;
	      w0r = (float) Math.cos (x1);
	      w0i = (float) Math.sin (x1);
	      w1r = 1.0f;
	      w1i = 0.0f;
//		----------------------------------------------------------------------------
	      for (int i = 1; i < kinc; i++) {
	        x1	= w0r * w1r - w0i * w1i;
	        w1i = w0r * w1i + w0i * w1r;
	        w1r = x1;
	        w2r = w1r * w1r - w1i * w1i;
	        w2i = w1r * w1i + w1i * w1r;
	        w3r = w2r * w1r - w2i * w1i;
	        w3i = w2r * w1i + w2i * w1r;
//		----------------------------------------------------------------------------
	        for (k0 = i; k0 < n; k0 += kinc2) {
	          k1			=  k0 + kinc;
	          k2			=  k1 + kinc;
	          k3			=  k2 + kinc;
	          rr			=  xr [k0] + xr [k2];
	          ri			=  xi [k0] + xi [k2];
	          sr			=  xr [k0] - xr [k2];
	          si			=  xi [k0] - xi [k2];
	          tr			=  xr [k1] + xr [k3];
	          ti			=  xi [k1] + xi [k3];
	          ur			= -xi [k1] + xi [k3];
	          ui			=  xr [k1] - xr [k3];
	          xr [k0]	=  rr + tr;
	          xi [k0] =  ri + ti;
//		----------------------------------------------------------------------------
	          qr = sr + ur;
	          qi = si + ui;
	          xr[k2] = qr * w1r - qi * w1i;
	          xi[k2] = qr * w1i + qi * w1r;
//		----------------------------------------------------------------------------
	          qr = rr - tr;
	          qi = ri - ti;
	          xr[k1] = qr * w2r - qi * w2i;
	          xi[k1] = qr * w2i + qi * w2r;
//		----------------------------------------------------------------------------
	          qr = sr - ur;
	          qi = si - ui;
	          xr[k3] = qr * w3r - qi * w3i;
	          xi[k3] = qr * w3i + qi * w3r;
	        }
	      }
	    }
//		----------------------------------------------------------------------------
//	  radix 2 section
//		----------------------------------------------------------------------------
	    while (kinc >= 2) {
	      kinc2	= kinc;
	      kinc	= kinc / 2;
	      x1		= (float) sTwoPi / (float) kinc2;
	      w0r		= (float) Math.cos (x1);
	      w0i		= (float) Math.sin (x1);
	      w1r		= 1.0f;
	      w1i		= 0.0f;
//		----------------------------------------------------------------------------
	      for (k0 = 0; k0 < n; k0 += kinc2) {
	        k1			= k0 + kinc;
	        tr			= xr [k0] - xr [k1];
	        ti			= xi [k0] - xi [k1];
	        xr [k0]	= xr [k0] + xr [k1];
	        xi [k0] = xi [k0] + xi [k1];
	        xr [k1] = tr;
	        xi [k1] = ti;
	      }
//		----------------------------------------------------------------------------
	      for (int i = 1; i < kinc; i++) {
	        x1	= w0r * w1r - w0i * w1i;
	        w1i = w0r * w1i + w0i * w1r;
	        w1r = x1;
//		----------------------------------------------------------------------------
	        for (k0 = i; k0 < n; k0 += kinc2) {
	          k1			= k0 + kinc;
	          tr			= xr [k0] - xr [k1];
	          ti			= xi [k0] - xi [k1];
	          xr [k0] = xr [k0] + xr [k1];
	          xi [k0] = xi [k0] + xi [k1];
	          xr [k1] = tr * w1r - ti * w1i;
	          xi [k1] = tr * w1i + ti * w1r;
	        }
	      }
	    }
//		----------------------------------------------------------------------------
//	  bit reverse order
//		----------------------------------------------------------------------------
	    int nv2 = n / 2;
	    int nm1 = n - 1;
	    int j = 0;
//		----------------------------------------------------------------------------
	    for (int i = 0; i < nm1; i++) {
//		----------------------------------------------------------------------------
	      if (i < j) {
	        tr			= xr [j];
	        ti			= xi [j];
	        xr [j]	= xr [i];
	        xi [j]	= xi [i];
	        xr [i]	= tr;
	        xi [i]	= ti;
	      }
	      k = nv2;
//		----------------------------------------------------------------------------
	      while (k <= j) {
	        j -= k;
	        k = k >> 1;
	      }
//		----------------------------------------------------------------------------
	      j += k;
	    }
//		----------------------------------------------------------------------------
	  }
}
